/* --------------------------------------------------------------
  TemplatePreview.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Template Preview Widget
 *
 * This widget is a wrapper for an iframe element that will display the template preview. Use
 * the "changePreview" method to change the current preview type.
 */
export default class TemplatePreview {
	/**
	 * Class Constructor 
	 * 
	 * @param {jQuery} $target Target <div> element of the template preview.
	 * @param {String} previewUrl The preview URL of the template.
	 */
	constructor($target, previewUrl) {
		StyleEdit.Validator.isObject($target);
		StyleEdit.Validator.isString(previewUrl);
		
		/**
		 * Target Preview Selector
		 *
		 * @type {jQuery}
		 */
		this.$target = $target;
		
		/**
		 * The template preview-URL
		 *
		 * @type {String}
		 */
		this.previewUrl = previewUrl;
	}
	
	/**
	 * Initialize the widget.
	 *
	 * Call this method to create the required HTML structure for the widget.
	 */
	initialize() {
		// Create the "loading" element. 
		const $loadingIcon = $($('#loading-spinner-template').html());
		
		$loadingIcon
			.css({
				display: 'table-cell',
				verticalAlign: 'middle',
				width: '60px'
			});
		
		const $loading = $('<div class="loading"></div>')
			.css({
				position: 'fixed',
				left: '0',
				top: '0',
				width: '100vw',
				height: '100vh',
				backgroundColor: '#FFF',
				fontSize: '60px',
				textAlign: 'center',
				display: 'table',
				overflow: 'hidden'
			});
		
		const $disableLayer = $('<div class="disable-layer"></div>')
			.css({
				display: 'none',
				position: 'fixed',
				left: '0',
				top: '0',
				width: '100%',
				height: '100%',
				backgroundColor: 'rgba(185, 74, 72, 0.27)',
				overflow: 'hidden'
			});
		
		const $iframe = $('<iframe src="' + this.previewUrl + '"></iframe>');
		
		this.$target.append($iframe);
		
		$('body')
			.append($disableLayer)
			.append($loading.append($loadingIcon));
		
		this._bindEventHandlers();
	}
	
	/**
	 * Change the current preview type of the template.
	 *
	 * @param {PreviewType} type The preview type to be used.
	 */
	changePreview(type) {
		StyleEdit.Validator.isObject(type);
		
		// Reload the iframe content to solve window.resize event problem.
		const iframeWindow = this.$target.find('iframe').get(0).contentWindow;
		
		if (iframeWindow.location.href !== 'about:blank') {
			iframeWindow.location.reload();
		}
		
		this.currentPreviewType = type;
	}
	
	/**
	 * Event: iframe Unload
	 *
	 * @private
	 */
	_onIframeUnload() {
		$('body').find('.loading').fadeIn();
	}
	
	/**
	 * Event: iframe Load Event
	 *
	 * Will be triggered once the iframe content has finished loading.
	 *
	 * @private
	 */
	_onIframeLoad() {
		this.$target.css(this.currentPreviewType.getPreviewCss());
		
		$('body').find('.loading').fadeOut();
		
		// Bind the unload event to the iframe window object.
		const iframeWindow = this.$target.find('iframe').get(0).contentWindow;
		$(iframeWindow).on('beforeunload', () => this._onIframeUnload());
	}
	
	/**
	 * Bind widget event handlers.
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$target.find('iframe')
			.off('load')
			.on('load', () => this._onIframeLoad());
	}
}
